﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Text;

namespace VA.PPMS.CRM.Plugins
{
    public static class PpmsHelper
    {
        #region Enumerations

        public enum LeieAction
        {
            Exclude = 767940000,
            Reinstate = 767940001
        }

        public enum AccountState
        {
            Active = 0,
            Inactive = 1
        }

        public enum Account_StatusCode
        {
            Active = 1,
            Inactive = 2,
            Pending = 767940000,
            Verified = 767940001,
            LeieExclusion = 767940002,
            AddressValidation = 767940004,
            NpiCheckFailure = 767940006,
            LicenseExpiration = 767940007,
            SamsExclusion = 767940008
        }

        public enum Batch_StatusCode
        {
            Active = 1,
            Inactive = 2,
            Processing = 767940000,
            ValidationComplete = 767940001,
            Resolved = 767940002
        }

        public enum CareSite_StatusCode
        {
            Active = 1,
            Inactive = 2,
            AddressValidationFailure = 767940000,
            LeieExclusion = 767940001
        }

        public enum ProviderService_StatusCode
        {
            Active = 1,
            Inactive = 2,
            AddressValidationFailure = 767940000
        }

        public enum Batch_TransType
        {
            Insert = 767940000,
            Update = 767940001,
            Delete = 767940002,
            LeieExclusion = 767940003
        }

        public enum Validation_Type
        {
            Leie = 767940000,
            License = 767940001,
            Address = 767940002,
            Geocode = 767940003,
            Nppes = 767940004,
            Contacts = 767940005,
            Sam = 767940006
        }

        public enum ProviderIdentifierType
        {
            NPI = 767940000,
            TIN = 767940001
        }

        public enum ProviderNpiType
        {
            Individual = 767940000,
            GroupAgency = 767940001,
            Organizational = 767940002
        }

        public enum HistoryLogType
        {
            Geocode = 767940003,
            Address = 767940006,
            Contact = 767940008,
            Leie = 767940004,
            NppesNpi = 767940005,
            License = 767940007,
            Sam = 767940009,
            CareSite = 767940010
        }

        #endregion

        #region Status Change Methods

        public static SetStateRequest GetDeactivateRequest(Entity entity, int statuscode = (int)Account_StatusCode.Inactive)
        {
            return GetDeactivateRequest(entity.ToEntityReference(), statuscode);
        }

        public static SetStateRequest GetDeactivateRequest(EntityReference entity, int statuscode = (int)Account_StatusCode.Inactive)
        {
            return GetStateRequest(entity, (int)AccountState.Inactive, statuscode);
        }

        public static SetStateRequest GetActivateRequest(Entity entity, int statuscode = (int)Account_StatusCode.Active)
        {
            return GetActivateRequest(entity.ToEntityReference(), statuscode);
        }

        public static SetStateRequest GetActivateRequest(EntityReference entity, int statuscode = (int)Account_StatusCode.Active)
        {
            return GetStateRequest(entity, (int)AccountState.Active, statuscode);
        }

        public static SetStateRequest GetStateRequest(Entity entity, int state, int statuscode)
        {
            return GetStateRequest(entity.ToEntityReference(), state, statuscode);
        }

        public static SetStateRequest GetStateRequest(EntityReference entity, int state, int statuscode)
        {
            return new SetStateRequest
            {
                EntityMoniker = entity,
                State = new OptionSetValue(state),
                Status = new OptionSetValue(statuscode)
            };
        }

        /// <summary>
        /// Get list of SetStateRequest objects based on the related entities
        /// </summary>
        /// <param name="entity">Parent entity</param>
        /// <param name="schemaName">Schema name</param>
        /// <param name="state">New state</param>
        /// <param name="statuscode">New status code</param>
        /// <returns>List of SetStateRequests</returns>
        public static IEnumerable<SetStateRequest> GetEntityStateRequests(Entity entity, string schemaName, int state, int statuscode)
        {
            List<SetStateRequest> requests = new List<SetStateRequest>();

            var children = entity.RelatedEntities[new Relationship(schemaName)];
            if (children != null)
            {
                foreach (var child in children.Entities)
                {
                    requests.Add(PpmsHelper.GetStateRequest(child, state, statuscode));
                }
            }

            return requests;
        }

        public static bool IsActive(Entity entity)
        {
            var statecode = entity.GetAttributeValue<OptionSetValue>("statecode");
            if (statecode == null || statecode.Value != (int)AccountState.Active)
                return false;
            else
                return true;
        }

        #endregion

        public static Entity GetProvider(IOrganizationService service, Entity npi)
        {
            var entity = npi.GetAttributeValue<EntityReference>("ppms_providerid");
            return GetProvider(service, entity);
        }

        public static Entity GetProvider(IOrganizationService service, EntityReference entity)
        {
            if (entity != null)
            {
                return service.Retrieve("account", entity.Id, new ColumnSet(new string[] { "accountid", "name", "statecode" }));
            }

            return null;
        }

        public static bool SetProviderState(IOrganizationService service, Entity provider, int stateValue)
        {
            var success = false;
            if (provider != null)
            {
                SetStateRequest request = null;
                switch (stateValue)
                {
                    case (int)PpmsHelper.LeieAction.Reinstate:
                        if (!PpmsHelper.IsActive(provider))
                        {
                            request = PpmsHelper.GetActivateRequest(provider);
                        }
                        break;
                    default:
                        if (PpmsHelper.IsActive(provider))
                        {
                            request = PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.LeieExclusion);
                        }
                        break;
                }

                // Execute request, if it has been set
                if (request != null)
                {
                    service.Execute(request);
                    success = true;
                }
            }
            return success;
        }

        public static string GetProviderComposite(Entity provider)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append(provider.GetAttributeValue<string>("name"));
            sb.Append(provider.GetAttributeValue<string>("address1_composite"));

            sb = sb.Replace("USA", String.Empty);
            sb = sb.Replace(Environment.NewLine, String.Empty);

            return sb.ToString().Trim();
        }

        public static string GetAddressComposite(Entity entity)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append(entity.GetAttributeValue<string>("ppms_address"));
            sb.Append(Environment.NewLine);
            sb.Append(entity.GetAttributeValue<string>("ppms_city"));
            sb.Append(", ");
            sb.Append(entity.GetAttributeValue<string>("ppms_state"));
            sb.Append(" ");
            sb.Append(entity.GetAttributeValue<string>("ppms_zip"));
            sb.Append("%");

            return sb.ToString().Trim();
        }

        public static int MapLeieActionToProviderState(int action)
        {
            int statecode = 0;

            if (action == (int)PpmsHelper.LeieAction.Reinstate)
            {
                statecode = (int)PpmsHelper.AccountState.Inactive;
            }
            else
            {
                statecode = (int)PpmsHelper.AccountState.Active;
            }
            return statecode;
        }

        public static void AddProviderValidation(IOrganizationService service, EntityReference provider, PpmsHelper.Validation_Type validationType)
        {
            //No longer creating validation entities. Update the Provider validation fields below. 
            // Add provider validation
            //var entity = new Entity("ppms_validation");
            //entity.Attributes.Add("ppms_name", validationType.ToString());
            //entity.Attributes.Add("ppms_validationtype", new OptionSetValue((int)validationType));
            //entity.Attributes.Add("ppms_providerid", provider);
            //service.Create(entity);


            //Switch here to define the validation field to update. 
            string validationField = "";
            switch (validationType)
            {
                case Validation_Type.Address:
                    validationField = "ppms_addressvalidated";
                    break;
                case Validation_Type.Contacts:
                    validationField = "ppms_providercontactvalidated";
                    break;
                case Validation_Type.Geocode:
                    validationField = "ppms_geocoded";
                    break;
                case Validation_Type.Leie:
                    validationField = "ppms_leievalidated";
                    break;
                case Validation_Type.License:
                    validationField = "ppms_licensevalidated";
                    break;
                case Validation_Type.Nppes:
                    validationField = "ppms_nppesnpivalidated";
                    break;
                case Validation_Type.Sam:
                    validationField = "ppms_samvalidated";
                    break;
            }

            var updateProv = new Entity("account");
            updateProv.Id = provider.Id;
            updateProv.Attributes.Add(validationField, true);
            service.Update(updateProv);           
        }
    }
}
